"""
========================================
🎭 VISUALモジュール - 縮小機能追加版 完全ガイド
========================================

pygame-zeroのActorクラスとpgzhelperを継承し、透明化・モザイク効果・色効果・回転機能・明るさ効果・拡大/縮小機能を統合したモジュールです。
一つのクラスで全ての見た目効果（透明、モザイク、色変更、回転、明るさ、拡大/縮小）+ 7つのアニメーションオブジェクト制御を提供します。

🔧 縮小機能追加版の特徴:
- 全エフェクト完全実装（透明・モザイク・色・回転・明るさ・拡大/縮小） ✅
- 🆕 縮小アニメーション対応（target < 1.0で自動判定） ✅
- 7つのアニメーションオブジェクト管理機能 ✅
- 一時停止・再開・停止機能付き ✅
- 個別アニメーション3つ + 同時アニメーション4つ ✅
- 既存コードとの完全互換性維持 ✅

📋 基本的な使用方法:
   from visual import VisualEffect
   
   # 全エフェクト付きActorを作成
   boon = VisualEffect("boon", (400, 300))
   
   def draw():
       boon.draw(screen)  # エフェクト付き自動描画
   
   def update():
       # 従来方式
       boon.process_animation()  # アニメーション処理
       
       # 新方式（アニメーションオブジェクト）
       animation.update()  # 個別制御

========================================
📂 主要プロパティ一覧
========================================

🔹 Actor継承プロパティ（そのまま使用可能）

   .x                       - X座標位置
   .y                       - Y座標位置
   .pos                     - 位置座標 (x, y)
   .visible                 - 表示状態フラグ
   .image                   - 画像ファイル名
   .angle                   - 回転角度（度数、0-359）
   .flip_x                  - 左右反転フラグ（pgzhelperのみ）
   .flip_y                  - 上下反転フラグ（pgzhelperのみ）

🔹 透明度関連プロパティ

   .alpha_level             - 現在の透明度（0-255）
   .alpha_increment         - 透明度の変化量（デフォルト10）
   .alpha_stop_at           - 透明度停止レベル（0-100%、デフォルト100）
   .is_alpha_on            - 透明度エフェクトのオン/オフ
   .animation_type         - アニメーションタイプ（"fade_out"/"fade_in"）

🔹 モザイク関連プロパティ

   .custom_pixel_size      - 現在のピクセルサイズ（初期値0）
   .pixel_increment        - 1段階あたりのピクセル増加量（デフォルト5）
   .current_mosaic_strength - 現在のモザイク強度（0-100）
   .is_mosaic_on           - モザイクエフェクトのオン/オフ
   .mosaic_stop_at         - モザイク停止レベル（％、デフォルト100）
   .max_pixel_size         - 最大ピクセルサイズ（自動計算）

🔹 色効果関連プロパティ

   .hue_shift              - 色相シフト値（0-199、200で一周）
   .base_bg_color          - 基本背景色（RGB タプル）
   .current_bg_color       - 現在の背景色（RGB タプル）
   .is_color_on            - 色効果のオン/オフ

🔹 明るさ関連プロパティ

   .brightness             - 明るさ値（-100～100）
   .is_brightness_on       - 明るさエフェクトのオン/オフ

🔹 拡大/縮小関連プロパティ（🆕 縮小対応）

   .scale_factor           - 拡大/縮小倍率（1.0=等倍）
   .scale_increment        - 拡大/縮小変化量（デフォルト0.05、自動で正負判定）
   .scale_target           - 拡大/縮小目標倍率（デフォルト3.0）
   .is_scaling             - 拡大/縮小アニメーション実行中

🔹 アニメーション関連プロパティ

   .animation_running      - アニメーション実行中フラグ
   .animation_speed        - アニメーション速度（フレーム数、デフォルト5）
   .animation_counter      - アニメーションカウンター

========================================
🛠️ メソッド一覧
========================================

🔸 初期化・設定メソッド

   VisualEffect(image, pos=(0, 0))
       説明: 全エフェクト付きActorを作成
       例: boon = VisualEffect("boon", (400, 300))

   .reset_effects()
       説明: 全てのエフェクトをリセット
       例: boon.reset_effects()

   .setup_image()
       説明: 画像を内部システムに設定・更新
       用途: 画像変更後にエフェクトを正しく適用するため
       例: boon.image = "ouch_boon"; boon.setup_image()

   .setup_visual_costume()
       説明: costumeモジュールとの連携用（画像同期）
       例: boon.setup_visual_costume()

🔸 従来アニメーションメソッド（process_animation使用）

   .alpha_animation()
       説明: フェードアウトアニメーション開始（不透明→透明）
       例: boon.alpha_animation()

   .fade_in_animation()
       説明: フェードインアニメーション開始（透明→不透明）
       例: boon.fade_in_animation()

   .mosaic_animation()
       説明: モザイクアニメーション開始
       例: boon.mosaic_animation()

   🆕 .scale_animation(increment=0.05, target=3.0)
       説明: 拡大/縮小アニメーション開始（自動判定）
       パラメータ: increment=変化量（常に正の値）、target=目標倍率
       動作: target > scale_factor なら拡大、target < scale_factor なら縮小
       例: 
           # 拡大アニメーション
           boon.scale_animation()                    # デフォルト（3倍に拡大）
           boon.scale_animation(0.1, 5.0)           # 高速で5倍に拡大
           boon.scale_animation(0.02, 1.5)          # ゆっくり1.5倍に拡大
           
           # 🆕 縮小アニメーション
           boon.scale_animation(0.05, 0.5)          # 0.5倍に縮小
           boon.scale_animation(0.02, 0.3)          # ゆっくり0.3倍に縮小
           boon.scale_animation(0.1, 0.1)           # 高速で0.1倍に縮小

   .process_animation()
       説明: アニメーション処理実行（従来方式、🆕 縮小対応）
       戻り値: "finished"=完了、True=継続中、None=未実行
       例: 
           result = boon.process_animation()
           if result == "finished":
               print("アニメーション完了！")

🔸 🆕 アニメーションオブジェクト作成メソッド（新機能）

【個別アニメーション】

   .create_fade_in_animation()
       説明: フェードインアニメーションオブジェクトを作成
       戻り値: FadeAnimationオブジェクト
       例: fade_in = boon.create_fade_in_animation()

   .create_fade_out_animation()
       説明: フェードアウトアニメーションオブジェクトを作成
       戻り値: FadeAnimationオブジェクト
       例: fade_out = boon.create_fade_out_animation()

   🆕 .create_scale_animation(increment=0.05, target=3.0)
       説明: 拡大/縮小アニメーションオブジェクトを作成
       パラメータ: increment=変化量、target=目標倍率
       戻り値: ScaleAnimationオブジェクト
       例: 
           # 拡大
           scale_up = boon.create_scale_animation(0.1, 4.0)
           
           # 🆕 縮小
           scale_down = boon.create_scale_animation(0.05, 0.5)

   .create_mosaic_animation()
       説明: モザイクアニメーションオブジェクトを作成
       戻り値: MosaicAnimationオブジェクト
       例: mosaic = boon.create_mosaic_animation()

【同時アニメーション】

   .create_mosaic_fade_animation(animation_type="fade_out")
       説明: モザイク+透明同時アニメーションオブジェクトを作成
       パラメータ: animation_type="fade_out"/"fade_in"
       戻り値: MosaicFadeAnimationオブジェクト
       例: 
           mosaic_fade_out = boon.create_mosaic_fade_animation("fade_out")
           mosaic_fade_in = boon.create_mosaic_fade_animation("fade_in")

   🆕 .create_scale_fade_animation(animation_type="fade_out", increment=0.05, target=3.0)
       説明: 拡大/縮小+透明同時アニメーションオブジェクトを作成
       パラメータ: animation_type、increment、target
       戻り値: ScaleFadeAnimationオブジェクト
       例: 
           # 拡大しながらフェードアウト
           scale_fade = boon.create_scale_fade_animation("fade_out", 0.1, 5.0)
           
           # 🆕 縮小しながらフェードアウト
           shrink_fade = boon.create_scale_fade_animation("fade_out", 0.05, 0.3)

   🆕 .create_mosaic_scale_animation(increment=0.05, target=3.0)
       説明: モザイク+拡大/縮小同時アニメーションオブジェクトを作成
       パラメータ: increment、target
       戻り値: MosaicScaleAnimationオブジェクト
       例: 
           # 拡大しながらモザイク
           mosaic_scale_up = boon.create_mosaic_scale_animation(0.08, 2.5)
           
           # 🆕 縮小しながらモザイク
           mosaic_scale_down = boon.create_mosaic_scale_animation(0.05, 0.5)

   🆕 .create_triple_animation(animation_type="fade_out", increment=0.05, target=3.0)
       説明: モザイク+透明+拡大/縮小同時アニメーションオブジェクトを作成
       パラメータ: animation_type、increment、target
       戻り値: TripleAnimationオブジェクト
       例: 
           # 拡大しながら透明＋モザイク
           triple = boon.create_triple_animation("fade_out", 0.05, 3.0)
           
           # 🆕 縮小しながら透明＋モザイク
           triple_shrink = boon.create_triple_animation("fade_out", 0.03, 0.2)

🔸 🆕 アニメーションオブジェクト共通メソッド（全7種対応）

   .start()
       説明: アニメーション開始
       戻り値: 自分自身（メソッドチェーン可能）
       例: 
           animation.start()
           animation = boon.create_fade_out_animation().start()  # チェーン可能

   .stop()
       説明: アニメーション停止（現在の状態で停止）
       戻り値: 自分自身
       例: animation.stop()

   .pause()
       説明: アニメーション一時停止
       戻り値: 自分自身
       例: animation.pause()

   .resume()
       説明: アニメーション再開
       戻り値: 自分自身
       例: animation.resume()

   .update()
       説明: アニメーション更新（毎フレーム呼び出し必須）
       戻り値: "finished"=完了、None=継続中/停止中
       例: 
           result = animation.update()
           if result == "finished":
               print("アニメーション完了！")

   .status()
       説明: 現在の状態を取得
       戻り値: "running"/"paused"/"stopped"/"completed"
       例: 
           current_status = animation.status()
           if current_status == "completed":
               print("完了しました")

🔸 即座効果メソッド

   .set_alpha(alpha_value)
       説明: 透明度を即座に設定（0-255）
       例: 
           boon.set_alpha(128)    # 半透明
           boon.set_alpha(0)      # 完全透明
           boon.set_alpha(255)    # 完全不透明

   .set_ghost(alpha_value)
       説明: set_alpha()の別名（互換性用）
       例: boon.set_ghost(100)

   .set_mosaic(strength)
       説明: モザイクの強さを即座に設定（0-100）
       例: 
           boon.set_mosaic(0)     # モザイクなし
           boon.set_mosaic(50)    # 中程度
           boon.set_mosaic(100)   # 最大

   🆕 .set_scale(scale_value)
       説明: 拡大/縮小倍率を即座に設定
       例: 
           boon.set_scale(2.0)    # 2倍に拡大
           boon.set_scale(0.5)    # 0.5倍に縮小（🆕）
           boon.set_scale(0.2)    # 0.2倍に縮小（🆕）
           boon.set_scale(1.0)    # 等倍に戻す

🔸 色効果メソッド

   .change_color(amount=25)
       説明: 色相を指定量変更（Scratchと同じ）
       例: 
           boon.change_color(20)    # 色相を20進める
           boon.change_color(-10)   # 色相を10戻す

   .set_color(value)
       説明: 色相を指定値に設定
       例: boon.set_color(100)

   .change_bg_color(amount=25)
       説明: 背景色の色相を変更
       例: 
           boon.change_bg_color(50)     # 背景色を大きく変更
           boon.change_bg_color(-25)    # 背景色を戻す

   .set_base_bg_color(color)
       説明: 背景色の基本色を設定
       例: 
           boon.set_base_bg_color((0, 100, 200))    # 青系
           boon.set_base_bg_color((255, 255, 255))  # 白

   .apply_bg_color(screen)
       説明: 現在の背景色でスクリーンを塗りつぶす
       例: boon.apply_bg_color(screen)

   .reset_color()
       説明: 色効果のみをリセット
       例: boon.reset_color()

   .get_current_color_effect()
       説明: 現在の色相シフト値を取得（0-199）
       例: current_shift = boon.get_current_color_effect()

   .get_bg_color()
       説明: 現在の背景色を取得（RGB タプル）
       例: current_color = boon.get_bg_color()

🔸 回転・移動メソッド

   .move_to(x, y)
       説明: 指定座標に移動
       例: boon.move_to(400, 300)

   .move_random(x_min=50, x_max=750, y_min=50, y_max=540)
       説明: ランダムな場所に移動
       例: 
           new_pos = boon.move_random()                    # デフォルト範囲
           new_pos = boon.move_random(0, 800, 0, 600)      # カスタム範囲

   .move_forward(distance)
       説明: 現在の角度方向に指定距離移動
       例: 
           boon.move_forward(10)    # 前に10ピクセル
           boon.move_forward(-5)    # 後ろに5ピクセル

🔸 明るさ効果メソッド

   .change_brightness(amount)
       説明: 明るさを指定量変更（Scratchと同じ）
       例: 
           boon.change_brightness(30)    # 30明るくする
           boon.change_brightness(-20)   # 20暗くする

   .set_brightness(value)
       説明: 明るさを指定値に設定（-100～100）
       例: 
           boon.set_brightness(50)      # 明るさを50に
           boon.set_brightness(-30)     # 明るさを-30に
           boon.set_brightness(0)       # 標準に戻す

   .reset_brightness()
       説明: 明るさをリセット
       例: boon.reset_brightness()

   .get_current_brightness()
       説明: 現在の明るさ値を取得（-100～100）
       例: current_brightness = boon.get_current_brightness()

🔸 描画・判定メソッド

   .draw(screen=None)
       説明: 全エフェクト付き画像を自動描画
       重要: screenパラメータを必ず渡すことを推奨
       例: boon.draw(screen)  # 推奨

   .collide_pixel(actor)
       説明: ピクセル単位の当たり判定
       例: 
           if boon.collide_pixel(enemy):
               print("当たった！")

========================================
🎪 実践的な使用例
========================================

# 🆕 縮小アニメーション使用例
def shrink_examples():
   # 基本的な縮小
   boon.scale_animation(0.05, 0.5)      # 0.5倍に縮小
   
   # 高速縮小
   boon.scale_animation(0.1, 0.2)       # 素早く小さく
   
   # ゆっくり縮小
   boon.scale_animation(0.01, 0.3)      # ゆっくり小さく
   
   # 縮小しながらフェードアウト
   shrink_fade = boon.create_scale_fade_animation("fade_out", 0.05, 0.3)
   shrink_fade.start()
   
   # 縮小しながらモザイク
   shrink_mosaic = boon.create_mosaic_scale_animation(0.05, 0.2)
   shrink_mosaic.start()

# 🆕 新方式：アニメーションオブジェクト使用
def new_animation_style():
   # アニメーションオブジェクト作成
   fade_out = boon.create_fade_out_animation()
   scale_up = boon.create_scale_animation(0.1, 4.0)
   scale_down = boon.create_scale_animation(0.05, 0.5)    # 🆕 縮小
   triple = boon.create_triple_animation("fade_in", 0.05, 2.0)
   
   # 制御
   scale_down.start()                  # 開始
   scale_down.pause()                  # 一時停止
   scale_down.resume()                 # 再開
   scale_down.stop()                   # 停止
   
   # 状態確認
   if scale_down.status() == "completed":
       print("縮小完了！")
   
   # 毎フレーム更新
   def update():
       result = scale_down.update()
       if result == "finished":
           print("アニメーション完了")

# 従来方式：process_animation使用（互換性維持）
def traditional_style():
   # 拡大アニメーション（従来通り）
   boon.scale_animation(0.1, 3.0)      # 拡大
   
   # 🆕 縮小アニメーション
   boon.scale_animation(0.05, 0.5)     # 縮小
   
   # 同時アニメーション
   boon.mosaic_animation()             # モザイク開始
   boon.alpha_animation()              # 透明開始
   boon.scale_animation(0.1, 0.3)      # 縮小開始（🆕）
   
   # 毎フレーム処理
   def update():
       result = boon.process_animation()
       if result == "finished":
           print("全アニメーション完了")

# 🎮 ゲームでの実用例
def game_usage_examples():
   # 敵撃破エフェクト（縮小版）
   enemy_death = enemy.create_scale_fade_animation("fade_out", 0.1, 0.1)  # 🆕
   enemy_death.start()
   
   # アイテム取得エフェクト（縮小）
   item_get = item.create_scale_fade_animation("fade_out", 0.2, 0.2)  # 🆕
   item_get.start()
   
   # プレイヤー復活エフェクト（拡大→通常）
   player.set_scale(0.1)                                                # 🆕 小さい状態から
   player_respawn = player.create_scale_animation(0.05, 1.0)           # 🆕 通常サイズに
   player_respawn.start()
   
   # ボス登場エフェクト（小→大）
   boss.set_scale(0.1)                                                  # 🆕 最小から
   boss_appear = boss.create_scale_animation(0.03, 1.0)                # 🆕 徐々に大きく
   boss_appear.start()
   
   # 敵が遠ざかる演出
   enemy_retreat = enemy.create_scale_animation(0.02, 0.3)             # 🆕 小さくなって消える
   enemy_retreat.start()

========================================
🔧 従来方式との比較
========================================

【従来方式（process_animation）】
- ✅ シンプルな記述
- ❌ 制御機能なし（停止・一時停止不可）
- ❌ 状態確認不可
- ✅ 複数エフェクト同時実行可能
- ✅ 🆕 拡大・縮小両対応

【新方式（アニメーションオブジェクト）】
- ✅ 完全制御機能（start/pause/resume/stop）
- ✅ 状態確認可能（status）
- ✅ メソッドチェーン対応
- ✅ 7種類のアニメーション組み合わせ
- ✅ 個別管理可能
- ✅ 🆕 拡大・縮小両対応

========================================
🆕 縮小機能の特徴
========================================

1. **自動判定システム**
   - target > scale_factor → 拡大アニメーション
   - target < scale_factor → 縮小アニメーション
   - incrementは常に正の値でOK

2. **既存コードへの影響なし**
   - 既存の拡大アニメーションはそのまま動作
   - 他のエフェクトには一切影響なし

3. **幅広い応用**
   - 敵が遠ざかる演出
   - アイテム取得時の縮小消滅
   - キャラクター復活時の拡大登場
   - ズームイン・ズームアウト効果

========================================
🎯 まとめ
========================================

縮小機能追加版visualモジュールは、従来の全機能を継承しつつ、
拡大・縮小の双方向アニメーションに対応した完全版です。

主な特徴:
- 🎭 6種類のエフェクト（透明・モザイク・色・回転・明るさ・拡大/縮小）
- 🎬 7つのアニメーションオブジェクト（個別3つ + 同時4つ）
- 🎮 完全制御機能（開始・停止・一時停止・再開）
- 🔄 従来方式との完全互換性
- ⚡ 🆕 拡大・縮小両対応（自動判定）
- 🛡️ 既存コードへの影響ゼロ

用途に応じて拡大・縮小を使い分けることで、
より豊かなゲーム演出を簡単に実現できます。

========================================
"""

import pygame
import random
import math
from pgzhelper import *

class FadeAnimation:
    """フェードアニメーションを管理するクラス"""
    
    def __init__(self, actor, animation_type="fade_in"):
        self.actor = actor
        self.animation_type = animation_type
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        if self.animation_type == "fade_in":
            self.actor.fade_in_animation()
        elif self.animation_type == "fade_out":
            self.actor.alpha_animation()
        
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.animation_running = False
        self.actor.animation_counter = 0
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self, 'animation_paused'):
            self.actor.animation_paused = False
        self.actor.animation_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新（毎フレーム呼び出し）"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class ScaleAnimation:
    """拡大/縮小アニメーションを管理するクラス"""
    
    def __init__(self, actor, increment=0.05, target=3.0):
        self.actor = actor
        self.increment = increment
        self.target = target
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        self.actor.scale_animation(self.increment, self.target)
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.is_scaling = False
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.actor.scale_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class MosaicAnimation:
    """モザイクアニメーションを管理するクラス"""
    
    def __init__(self, actor):
        self.actor = actor
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        self.actor.mosaic_animation()
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.animation_running = False
        self.actor.animation_counter = 0
        self.actor.is_mosaic_on = False
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.actor.animation_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class MosaicFadeAnimation:
    """モザイク+透明同時アニメーションを管理するクラス"""
    
    def __init__(self, actor, animation_type="fade_out"):
        self.actor = actor
        self.animation_type = animation_type
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        if self.animation_type == "fade_in":
            self.actor.fade_in_animation()
        else:
            self.actor.alpha_animation()
        
        self.actor.mosaic_animation()
        
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.animation_running = False
        self.actor.animation_counter = 0
        self.actor.is_mosaic_on = False
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.actor.animation_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class ScaleFadeAnimation:
    """拡大/縮小+透明同時アニメーションを管理するクラス"""
    
    def __init__(self, actor, animation_type="fade_out", increment=0.05, target=3.0):
        self.actor = actor
        self.animation_type = animation_type
        self.increment = increment
        self.target = target
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        self.actor.scale_animation(self.increment, self.target)
        self.actor.is_alpha_on = True
        self.actor.animation_type = self.animation_type
        
        if self.animation_type == "fade_in":
            self.actor.fade_in_animation()
        else:
            self.actor.alpha_animation()
        
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.animation_running = False
        self.actor.animation_counter = 0
        self.actor.is_scaling = False
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.actor.animation_paused = True
        if not hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.actor.scale_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        if hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class MosaicScaleAnimation:
    """モザイク+拡大/縮小同時アニメーションを管理するクラス"""
    
    def __init__(self, actor, increment=0.05, target=3.0):
        self.actor = actor
        self.increment = increment
        self.target = target
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        self.actor.scale_animation(self.increment, self.target)
        self.actor.mosaic_animation()
        
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.animation_running = False
        self.actor.animation_counter = 0
        self.actor.is_scaling = False
        self.actor.is_mosaic_on = False
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.actor.animation_paused = True
        if not hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.actor.scale_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        if hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class TripleAnimation:
    """モザイク+透明+拡大/縮小同時アニメーションを管理するクラス"""
    
    def __init__(self, actor, animation_type="fade_out", increment=0.05, target=3.0):
        self.actor = actor
        self.animation_type = animation_type
        self.increment = increment
        self.target = target
        self.is_running = False
        self.is_paused = False
        self.is_completed = False
        
    def start(self):
        """アニメーション開始"""
        self.actor.scale_animation(self.increment, self.target)
        
        if self.animation_type == "fade_in":
            self.actor.fade_in_animation()
        else:
            self.actor.alpha_animation()
        
        self.actor.mosaic_animation()
        
        self.is_running = True
        self.is_paused = False
        self.is_completed = False
        return self
    
    def stop(self):
        """アニメーション停止"""
        self.actor.animation_running = False
        self.actor.animation_counter = 0
        self.actor.is_scaling = False
        self.is_running = False
        self.is_paused = False
        return self
    
    def pause(self):
        """アニメーション一時停止"""
        if not hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        self.actor.animation_paused = True
        if not hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.actor.scale_paused = True
        self.is_paused = True
        return self
    
    def resume(self):
        """アニメーション再開"""
        if hasattr(self.actor, 'animation_paused'):
            self.actor.animation_paused = False
        if hasattr(self.actor, 'scale_paused'):
            self.actor.scale_paused = False
        self.is_paused = False
        return self
    
    def update(self):
        """アニメーション更新"""
        if self.is_running and not self.is_paused:
            result = self.actor.process_animation()
            if result == "finished":
                self.is_running = False
                self.is_completed = True
                return "finished"
        return None
    
    def status(self):
        """現在の状態を取得"""
        if self.is_completed:
            return "completed"
        elif self.is_running and self.is_paused:
            return "paused"
        elif self.is_running:
            return "running"
        else:
            return "stopped"

class VisualEffect(Actor):
    
    def _inherit_say_functionality(self):
        try:
            from say import text_display
            
            def say(text, seconds=None, size=40, color="black", x_offset=None, y_offset=None, fontname=None, anchor="center"):
                text_display.say(self, text, seconds, size, color, x_offset, y_offset, fontname, anchor)
            
            def is_talking():
                return text_display.is_talking(self)
            
            def clear_text():
                text_display.clear(self)
            
            self.say = say
            self.is_talking = is_talking
            self.clear_text = clear_text
            
        except ImportError:
            pass
    
    def __init__(self, *args, **kwargs):
        if len(args) == 1:
            super().__init__(args[0], (0, 0))
        elif len(args) == 2:
            super().__init__(args[0], args[1])
        else:
            super().__init__(*args, **kwargs)
        
        self.visible = True
        
        # 回転属性
        self.angle = 0.0
        self._angle = 0.0
        
        # 明るさ属性
        self.brightness = 0
        self.is_brightness_on = False
        
        # 拡大属性
        self.scale_factor = 1.0
        self.scale_increment = 0.05
        self.scale_target = 3.0
        self.is_scaling = False
        
        # sayモジュール機能継承
        self._inherit_say_functionality()
        
        # エフェクト用属性
        self.original_image = None
        self.processed_image = None
        self.current_looks_actor = None
        
        # 透明度関連
        self.alpha_level = 255
        self.alpha_increment = 10
        self.alpha_stop_at = 100
        self.is_alpha_on = False
        
        # モザイク関連
        self.is_pixelated = False
        self.pixelation_level = 0
        self.pixel_increment = 5
        self.custom_pixel_size = 0
        self.is_mosaic_on = False
        self.mosaic_stop_at = 100
        self.current_mosaic_strength = 0
        
        # 色効果関連
        self.hue_shift = 0
        self.base_bg_color = (255, 255, 255)
        self.current_bg_color = self.base_bg_color
        self.is_color_on = False
        
        # アニメーション関連
        self.animation_type = "fade_out"
        self.animation_running = False
        self.animation_speed = 5
        self.animation_counter = 0
        
        # 表示状態
        self.is_visible = True
        self.is_actor_draw = True
        
        # 初期化
        self.setup_image()

    def setup_image(self):
        """画像を内部システムに設定・更新する"""
        if hasattr(self, '_surf') and self._surf:
            self.original_image = self._surf.copy()
            self.processed_image = self._surf.copy()
            width, height = self._surf.get_size()
            self.max_pixel_size = min(width, height)

    # アニメーションオブジェクト作成メソッド
    def create_fade_in_animation(self):
        """フェードインアニメーションオブジェクトを作成"""
        return FadeAnimation(self, "fade_in")

    def create_fade_out_animation(self):
        """フェードアウトアニメーションオブジェクトを作成"""
        return FadeAnimation(self, "fade_out")

    def create_scale_animation(self, increment=0.05, target=3.0):
        """拡大/縮小アニメーションオブジェクトを作成"""
        return ScaleAnimation(self, increment, target)

    def create_mosaic_animation(self):
        """モザイクアニメーションオブジェクトを作成"""
        return MosaicAnimation(self)

    def create_mosaic_fade_animation(self, animation_type="fade_out"):
        """モザイク+透明同時アニメーションオブジェクトを作成"""
        return MosaicFadeAnimation(self, animation_type)

    def create_scale_fade_animation(self, animation_type="fade_out", increment=0.05, target=3.0):
        """拡大/縮小+透明同時アニメーションオブジェクトを作成"""
        return ScaleFadeAnimation(self, animation_type, increment, target)

    def create_mosaic_scale_animation(self, increment=0.05, target=3.0):
        """モザイク+拡大/縮小同時アニメーションオブジェクトを作成"""
        return MosaicScaleAnimation(self, increment, target)

    def create_triple_animation(self, animation_type="fade_out", increment=0.05, target=3.0):
        """モザイク+透明+拡大/縮小同時アニメーションオブジェクトを作成"""
        return TripleAnimation(self, animation_type, increment, target)

    def _update_current_drawable(self):
        has_effects = (self.is_mosaic_on or self.is_alpha_on or 
                      self.is_color_on or self.is_brightness_on or 
                      self.scale_factor != 1.0)
        
        if has_effects and self.processed_image:
            self.current_looks_actor = self.processed_image
            self._angle = self.angle
        else:
            self.current_looks_actor = None
            self._angle = self.angle

    def _update_draw_position(self):
        if self.processed_image:
            self.draw_pos = (
                self.x - self.processed_image.get_width() // 2,
                self.y - self.processed_image.get_height() // 2
            )

    def move_to(self, x, y):
        self.x = x
        self.y = y

    def move_random(self, x_min=50, x_max=750, y_min=50, y_max=540):
        self.x = random.randint(x_min, x_max)
        self.y = random.randint(y_min, y_max)
        return (self.x, self.y)
    
    def move_forward(self, distance):
        angle_rad = math.radians(self.angle)
        self.x += distance * math.cos(angle_rad)
        self.y -= distance * math.sin(angle_rad)

    def scale_animation(self, increment=0.05, target=3.0):
        """🆕 拡大/縮小アニメーション開始（縮小対応版）"""
        if not self.original_image:
            return False
        
        # 🆕 拡大・縮小を自動判定
        if target < self.scale_factor:
            # 縮小の場合は負の値に
            self.scale_increment = -abs(increment)
        else:
            # 拡大の場合は正の値に
            self.scale_increment = abs(increment)
        
        self.scale_target = target
        self.is_scaling = True
        return True

    def change_brightness(self, amount):
        self.brightness = max(-100, min(100, self.brightness + amount))
        self.is_brightness_on = True if self.brightness != 0 else False
        self._apply_all_effects()
        return self.brightness
        
    def set_brightness(self, value):
        self.brightness = max(-100, min(100, value))
        self.is_brightness_on = True if self.brightness != 0 else False
        self._apply_all_effects()
        return self.brightness
        
    def reset_brightness(self):
        self.brightness = 0
        self.is_brightness_on = False
        self._apply_all_effects()
        
    def get_current_brightness(self):
        return self.brightness

    def alpha_animation(self):
        if not self.original_image:
            return False
        
        self.is_alpha_on = True
        self.animation_type = "fade_out"
        
        if not self.animation_running:
            self.start_animation()

    def mosaic_animation(self):
        if not self.original_image:
            return False
        
        self.is_mosaic_on = True
        self.animation_type = "fade_out"
        
        if not self.animation_running:
            self.start_animation()

    def set_mosaic(self, strength):
        strength = max(0, min(100, int(strength)))
        self.current_mosaic_strength = strength
        
        if strength > 0:
            self.is_mosaic_on = True
        else:
            self.is_mosaic_on = False
        
        self._apply_all_effects()
        return strength

    def fade_in_animation(self):
        if not self.original_image:
            return False
        
        self.animation_type = "fade_in"
        self.animation_running = True
        self.animation_counter = 0
        
        self.alpha_level = 0
        self.is_alpha_on = True
        
        if self.original_image:
            self.processed_image = self.original_image.copy()
            self.processed_image.set_alpha(0)
            self._update_current_drawable()
        
        self.visible = True
        self.is_actor_draw = True
        
        return True

    def start_animation(self):
        self.animation_running = True
        self.pixelation_level = 1
        self.custom_pixel_size = 5
        
        if self.animation_type == "fade_in":
            self.alpha_level = 0
        else:
            self.alpha_level = 255
            
        self.animation_counter = 0
        
        if self.original_image:
            self.processed_image = self.original_image.copy()
            self._apply_all_effects()
            
        self._update_current_drawable()
        self.visible = True
        self.is_actor_draw = True

    def process_animation(self):
        """🆕 拡大/縮小対応版アニメーション処理"""
        # 一時停止中は処理をスキップ
        if hasattr(self, 'animation_paused') and self.animation_paused:
            return None
        
        result = None
        alpha_mosaic_finished = False

        # 🆕 拡大/縮小処理（両対応）
        scale_finished = False
        if self.is_scaling:
            # 一時停止チェック
            if hasattr(self, 'scale_paused') and self.scale_paused:
                pass
            else:
                saved_angle = self.angle
                self.scale_factor += self.scale_increment
                
                # 🆕 目標到達チェック（拡大・縮小共通）
                if (self.scale_increment > 0 and self.scale_factor >= self.scale_target) or \
                   (self.scale_increment < 0 and self.scale_factor <= self.scale_target):
                    self.scale_factor = self.scale_target
                    self.is_scaling = False
                    scale_finished = True
                
                self.angle = saved_angle
                self._angle = saved_angle

        if self.animation_running:
            self.animation_counter += 1

            if self.animation_counter % self.animation_speed == 0:
                if self.animation_type == "fade_in":
                    if self.alpha_level < 255:
                        self.alpha_level = min(255, self.alpha_level + self.alpha_increment)
                        
                        if self.original_image:
                            self.processed_image = self.original_image.copy()
                            
                            if self.is_color_on and self.hue_shift != 0:
                                self.processed_image = self._apply_color_shift(self.processed_image)
                            if self.is_brightness_on and self.brightness != 0:
                                self.processed_image = self._apply_brightness_effect(self.processed_image)
                            if self.scale_factor != 1.0:
                                original_size = self.processed_image.get_size()
                                new_width = int(original_size[0] * self.scale_factor)
                                new_height = int(original_size[1] * self.scale_factor)
                                self.processed_image = pygame.transform.scale(self.processed_image, (new_width, new_height))
                            
                            self.processed_image.set_alpha(self.alpha_level)
                            self._update_current_drawable()
                            
                        self.visible = True
                        self.is_actor_draw = True
                    
                    if self.alpha_level >= 255:
                        self.animation_running = False
                        self.is_alpha_on = False
                        self.is_actor_draw = True
                        self.visible = True
                        self.current_looks_actor = None
                        result = "finished"
                        alpha_mosaic_finished = True
                else:
                    result = self.update_animation()
                    
                    if result == "finished":
                        alpha_mosaic_finished = True
                        self.is_mosaic_on = False
                        self.is_alpha_on = False
                        
                        if self.animation_type == "fade_out":
                            self.is_actor_draw = False
                            self.visible = False
                        else:
                            self.is_actor_draw = True
                            self.visible = True
                        
                        self.current_looks_actor = None
                        self._update_current_drawable()

        if alpha_mosaic_finished or scale_finished:
            self.scale_factor = 1.0
            return "finished"
        else:
            return result

    def update_animation(self):
        if not self.animation_running:
            return False

        if self.original_image:
            self.processed_image = self.original_image.copy()
        
        if self.is_mosaic_on:
            self.pixelation_level += 1
            self.custom_pixel_size = self.pixel_increment * self.pixelation_level
            self.processed_image = self.pixelate(self.processed_image, self.custom_pixel_size)
        
        if self.is_color_on and self.hue_shift != 0:
            self.processed_image = self._apply_color_shift(self.processed_image)
        
        if self.is_brightness_on and self.brightness != 0:
            self.processed_image = self._apply_brightness_effect(self.processed_image)
        
        if self.scale_factor != 1.0:
            original_size = self.processed_image.get_size()
            new_width = int(original_size[0] * self.scale_factor)
            new_height = int(original_size[1] * self.scale_factor)
            self.processed_image = pygame.transform.scale(self.processed_image, (new_width, new_height))
        
        if self.is_alpha_on:
            if self.animation_type == "fade_out":
                target_alpha = int(255 * (100 - self.alpha_stop_at) / 100)
                self.alpha_level = max(target_alpha, self.alpha_level - self.alpha_increment)
            else:
                self.alpha_level = min(255, self.alpha_level + self.alpha_increment)
            
            self.processed_image.set_alpha(self.alpha_level)
        
        self._update_draw_position()
        self._update_current_drawable()
        
        if self.is_alpha_on:
            if self.animation_type == "fade_out":
                target_alpha = int(255 * (100 - self.alpha_stop_at) / 100)
                if self.alpha_level <= target_alpha:
                    self.animation_running = False
                    if self.animation_type == "fade_out":
                        self.is_visible = False
                        self.visible = False
                    return "finished"
            else:
                if self.alpha_level >= 255:
                    self.animation_running = False
                    return "finished"
        
        mosaic_stop_strength = (self.mosaic_stop_at / 100) * self.max_pixel_size
        if self.is_mosaic_on and self.custom_pixel_size >= mosaic_stop_strength:
            self.animation_running = False
            if self.animation_type == "fade_out":
                self.is_visible = False
                self.visible = False
            return "finished"

        return True

    def pixelate(self, surface, pixel_size):
        try:
            pixelated_surface = pygame.Surface(surface.get_size(), pygame.SRCALPHA)
            width, height = surface.get_size()

            for y in range(0, height, pixel_size):
                for x in range(0, width, pixel_size):
                    block_width = min(pixel_size, width - x)
                    block_height = min(pixel_size, height - y)

                    if block_width <= 0 or block_height <= 0:
                        continue

                    rect = pygame.Rect(x, y, block_width, block_height)
                    sub_surface = surface.subsurface(rect)
                    avg_color = pygame.transform.average_color(sub_surface)
                    pixelated_surface.fill(avg_color, rect=(x, y, block_width, block_height))

            return pixelated_surface
        except Exception as e:
            return surface.copy()

    def set_alpha(self, alpha_value):
        self.alpha_level = max(0, min(255, int(alpha_value)))
        if self.alpha_level < 255:
            self.is_alpha_on = True
        else:
            self.is_alpha_on = False
        self._apply_all_effects()
        if self.is_alpha_on:
            self.visible = False
        elif not (self.is_mosaic_on or self.is_color_on or self.is_brightness_on):
            self.visible = True
    
    def set_ghost(self, alpha_value):
        return self.set_alpha(alpha_value)

    def set_scale(self, scale_value):
        self.scale_factor = max(0.1, scale_value)
        self._apply_all_effects()
        return self.scale_factor

    def change_color(self, amount=25):
        self.hue_shift = (self.hue_shift + amount) % 200
        self.is_color_on = True if self.hue_shift != 0 else False
        self._apply_all_effects()
        return self.hue_shift
    
    def set_color(self, value):
        self.hue_shift = value % 200
        self.is_color_on = True if self.hue_shift != 0 else False
        self._apply_all_effects()
        return self.hue_shift
    
    def change_bg_color(self, amount=25):
        self.hue_shift = (self.hue_shift + amount) % 200
        self.is_color_on = True if self.hue_shift != 0 else False
        self._update_bg_color()
        return self.hue_shift
    
    def set_base_bg_color(self, color):
        self.base_bg_color = color
        self.current_bg_color = color
    
    def _update_bg_color(self):
        if self.hue_shift != 0:
            r, g, b = self.base_bg_color
            h, s, v = self._rgb_to_hsv(r/255, g/255, b/255)
            h = (h + self.hue_shift / 200) % 1.0
            if s < 0.3:
                s = 0.7
            r, g, b = self._hsv_to_rgb(h, s, v)
            self.current_bg_color = (int(r * 255), int(g * 255), int(b * 255))
        else:
            self.current_bg_color = self.base_bg_color
    
    def get_bg_color(self):
        return self.current_bg_color
    
    def apply_bg_color(self, screen):
        self._update_bg_color()
        screen.fill(self.current_bg_color)
    
    def reset_color(self):
        self.hue_shift = 0
        self.current_bg_color = self.base_bg_color
        self.is_color_on = False
        self._apply_all_effects()
    
    def get_current_color_effect(self):
        return self.hue_shift

    def reset_effects(self):
        if self.original_image:
            self.processed_image = self.original_image.copy()
            self.is_pixelated = False
            self.pixelation_level = 0
            self.is_visible = True
            self.custom_pixel_size = 0
            self.alpha_level = 255
            self.animation_running = False
            self.animation_counter = 0
            self.is_mosaic_on = False
            self.is_alpha_on = False
            self.hue_shift = 0
            self.is_color_on = False
            self.current_bg_color = self.base_bg_color
            self.angle = 0.0
            self.brightness = 0
            self.is_brightness_on = False
            self.scale_factor = 1.0
            self.is_scaling = False
            self.pixelation_level = 0
            self.current_mosaic_strength = 0
            self.is_actor_draw = True
            self.animation_type = "fade_out"
            self.visible = True
            self.current_looks_actor = None
            self._update_current_drawable()

    def _apply_all_effects(self):
        if not self.original_image:
            return
        
        base_image = self.original_image.copy()
        
        if self.is_color_on and self.hue_shift != 0:
            base_image = self._apply_color_shift(base_image)
        
        if self.is_brightness_on and self.brightness != 0:
            base_image = self._apply_brightness_effect(base_image)
        
        if self.is_mosaic_on and not self.animation_running:
            pixel_size = max(1, int(self.max_pixel_size * (self.current_mosaic_strength / 100)))
            base_image = self.pixelate(base_image, pixel_size)
        
        if self.scale_factor != 1.0:
            base_image = pygame.transform.scale(base_image, 
                (int(base_image.get_width() * self.scale_factor),
                 int(base_image.get_height() * self.scale_factor)))
        
        if self.is_alpha_on:
            base_image.set_alpha(self.alpha_level)
        
        self.processed_image = base_image
        self._update_current_drawable()

    def _apply_color_shift(self, surface):
        """色相シフト効果の適用"""
        if self.hue_shift == 0:
            return surface
        
        try:
            new_surf = surface.copy()
            w, h = surface.get_size()
            
            shift_amount = self.hue_shift / 200.0
            
            for x in range(w):
                for y in range(h):
                    try:
                        pixel = surface.get_at((x, y))
                        if len(pixel) >= 4:
                            r, g, b, a = pixel
                            if a > 0:
                                h_norm, s, v = self._rgb_to_hsv(r/255, g/255, b/255)
                                h_norm = (h_norm + shift_amount) % 1.0
                                r, g, b = self._hsv_to_rgb(h_norm, s, v)
                                new_surf.set_at((x, y), (int(r*255), int(g*255), int(b*255), a))
                        else:
                            r, g, b = pixel[:3]
                            h_norm, s, v = self._rgb_to_hsv(r/255, g/255, b/255)
                            h_norm = (h_norm + shift_amount) % 1.0
                            r, g, b = self._hsv_to_rgb(h_norm, s, v)
                            new_surf.set_at((x, y), (int(r*255), int(g*255), int(b*255)))
                    except (IndexError, ValueError):
                        continue
            
            return new_surf
        except Exception:
            return surface

    def _apply_brightness_effect(self, surface):
        """明るさ効果の適用"""
        if self.brightness == 0:
            return surface
        
        try:
            new_surf = surface.copy()
            w, h = surface.get_size()
            
            if self.brightness > 0:
                factor = self.brightness / 100.0
                for x in range(w):
                    for y in range(h):
                        try:
                            pixel = surface.get_at((x, y))
                            if len(pixel) >= 4:
                                r, g, b, a = pixel
                                if a > 0:
                                    r = min(255, int(r + (255 - r) * factor))
                                    g = min(255, int(g + (255 - g) * factor))
                                    b = min(255, int(b + (255 - b) * factor))
                                    new_surf.set_at((x, y), (r, g, b, a))
                            else:
                                r, g, b = pixel[:3]
                                r = min(255, int(r + (255 - r) * factor))
                                g = min(255, int(g + (255 - g) * factor))
                                b = min(255, int(b + (255 - b) * factor))
                                new_surf.set_at((x, y), (r, g, b))
                        except (IndexError, ValueError):
                            continue
            else:
                factor = abs(self.brightness) / 100.0
                for x in range(w):
                    for y in range(h):
                        try:
                            pixel = surface.get_at((x, y))
                            if len(pixel) >= 4:
                                r, g, b, a = pixel
                                if a > 0:
                                    r = max(0, int(r * (1 - factor)))
                                    g = max(0, int(g * (1 - factor)))
                                    b = max(0, int(b * (1 - factor)))
                                    new_surf.set_at((x, y), (r, g, b, a))
                            else:
                                r, g, b = pixel[:3]
                                r = max(0, int(r * (1 - factor)))
                                g = max(0, int(g * (1 - factor)))
                                b = max(0, int(b * (1 - factor)))
                                new_surf.set_at((x, y), (r, g, b))
                        except (IndexError, ValueError):
                            continue
            
            return new_surf
        except Exception:
            return surface

    def setup_visual_costume(self):
        """costumeモジュールとの連携用メソッド：画像変更時にvisualエフェクトを同期"""
        if hasattr(self, '_surf') and self._surf:
            saved_angle = getattr(self, 'angle', 0.0)
            saved_flip_x = getattr(self, 'flip_x', False)
            saved_flip_y = getattr(self, 'flip_y', False)
            saved_pos = (self.x, self.y)
            
            current_mosaic_strength = self.current_mosaic_strength
            current_alpha_level = self.alpha_level
            current_alpha_on = self.is_alpha_on
            current_color_on = self.is_color_on
            current_hue_shift = self.hue_shift
            current_brightness = self.brightness
            current_brightness_on = self.is_brightness_on
            current_scale_factor = self.scale_factor
            current_is_scaling = self.is_scaling
            
            current_animation_running = self.animation_running
            current_animation_type = self.animation_type
            current_animation_counter = self.animation_counter
            
            self.setup_image()
            
            self.angle = saved_angle
            self._angle = saved_angle
            if hasattr(self, 'flip_x'):
                self.flip_x = saved_flip_x
            if hasattr(self, 'flip_y'):
                self.flip_y = saved_flip_y
            self.x, self.y = saved_pos
            
            self.current_mosaic_strength = current_mosaic_strength
            self.alpha_level = current_alpha_level
            self.is_alpha_on = current_alpha_on
            self.is_color_on = current_color_on
            self.hue_shift = current_hue_shift
            self.brightness = current_brightness
            self.is_brightness_on = current_brightness_on
            self.scale_factor = current_scale_factor
            self.is_scaling = current_is_scaling
            
            self.animation_running = current_animation_running
            self.animation_type = current_animation_type
            self.animation_counter = current_animation_counter
            
            if (self.current_mosaic_strength > 0 or self.is_alpha_on or 
                self.is_color_on or self.is_brightness_on or 
                self.scale_factor != 1.0):
                self._apply_all_effects()
            
            self._update_current_drawable()

    def collide_pixel(self, actor):
        """完全互換のcollide_pixel()メソッド"""
        try:
            if hasattr(self, 'alpha_level') and self.alpha_level < 255:
                if hasattr(self, 'original_image') and self.original_image:
                    temp_self_surface = self.original_image.copy()
                    temp_self_surface.set_alpha(self.alpha_level)
                else:
                    temp_self_surface = self._surf
            else:
                temp_self_surface = self._surf
            
            if hasattr(actor, 'alpha_level') and hasattr(actor, 'original_image'):
                if actor.alpha_level < 255 and actor.original_image:
                    temp_actor_surface = actor.original_image.copy()
                    temp_actor_surface.set_alpha(actor.alpha_level)
                else:
                    temp_actor_surface = actor._surf
            else:
                temp_actor_surface = actor._surf
            
            self_rect = temp_self_surface.get_rect(center=(int(self.x), int(self.y)))
            actor_rect = temp_actor_surface.get_rect(center=(int(actor.x), int(actor.y)))
            
            if not self_rect.colliderect(actor_rect):
                return False
            
            overlap_rect = self_rect.clip(actor_rect)
            
            self_offset = (overlap_rect.left - self_rect.left, overlap_rect.top - self_rect.top)
            actor_offset = (overlap_rect.left - actor_rect.left, overlap_rect.top - actor_rect.top)
            
            for y in range(overlap_rect.height):
                for x in range(overlap_rect.width):
                    self_x, self_y = self_offset[0] + x, self_offset[1] + y
                    actor_x, actor_y = actor_offset[0] + x, actor_offset[1] + y
                    
                    if (0 <= self_x < temp_self_surface.get_width() and 
                        0 <= self_y < temp_self_surface.get_height() and
                        0 <= actor_x < temp_actor_surface.get_width() and 
                        0 <= actor_y < temp_actor_surface.get_height()):
                        
                        try:
                            self_pixel = temp_self_surface.get_at((self_x, self_y))
                            actor_pixel = temp_actor_surface.get_at((actor_x, actor_y))
                            
                            if len(self_pixel) >= 4 and len(actor_pixel) >= 4:
                                if self_pixel[3] > 0 and actor_pixel[3] > 0:
                                    return True
                            else:
                                return True
                        except (IndexError, ValueError):
                            continue
            
            return False
            
        except Exception as e:
            return super().colliderect(actor)

    def draw(self, screen=None):
        """エフェクト適用済み画像の描画"""
        if not self.visible or not self.is_actor_draw:
            return
        
        if (self.is_brightness_on and self.brightness != 0 and 
            not self.is_alpha_on and not self.is_mosaic_on and 
            not self.is_color_on and self.scale_factor == 1.0):
            
            if self.processed_image:
                image_to_draw = self.processed_image
                
                if hasattr(self, 'flip_x') and self.flip_x:
                    image_to_draw = pygame.transform.flip(image_to_draw, True, False)
                if hasattr(self, 'flip_y') and self.flip_y:
                    image_to_draw = pygame.transform.flip(image_to_draw, False, True)
                
                if self.angle != 0:
                    image_to_draw = pygame.transform.rotate(image_to_draw, self.angle)
                
                draw_x = self.x - image_to_draw.get_width() // 2
                draw_y = self.y - image_to_draw.get_height() // 2
                
                if screen:
                    screen.blit(image_to_draw, (draw_x, draw_y))
                else:
                    import __main__
                    if hasattr(__main__, 'screen'):
                        __main__.screen.blit(image_to_draw, (draw_x, draw_y))
                return
        
        if self.is_mosaic_on or self.is_alpha_on or self.is_color_on or self.is_brightness_on or self.scale_factor != 1.0:
            if self.processed_image:
                image_to_draw = self.processed_image
                
                if hasattr(self, 'flip_x') and self.flip_x:
                    image_to_draw = pygame.transform.flip(image_to_draw, True, False)
                if hasattr(self, 'flip_y') and self.flip_y:
                    image_to_draw = pygame.transform.flip(image_to_draw, False, True)
                
                if self.angle != 0:
                    image_to_draw = pygame.transform.rotate(image_to_draw, self.angle)
                
                draw_x = self.x - image_to_draw.get_width() // 2
                draw_y = self.y - image_to_draw.get_height() // 2
                
                if screen:
                    screen.blit(image_to_draw, (draw_x, draw_y))
                else:
                    import __main__
                    if hasattr(__main__, 'screen'):
                        __main__.screen.blit(image_to_draw, (draw_x, draw_y))
            else:
                super().draw()
        else:
            super().draw()

    def _rgb_to_hsv(self, r, g, b):
        mx = max(r, g, b)
        mn = min(r, g, b)
        df = mx - mn
        if mx == mn:
            h = 0
        elif mx == r:
            h = (60 * ((g - b) / df) + 360) % 360
        elif mx == g:
            h = (60 * ((b - r) / df) + 120) % 360
        elif mx == b:
            h = (60 * ((r - g) / df) + 240) % 360
        s = 0 if mx == 0 else df / mx
        v = mx
        return h / 360, s, v
    
    def _hsv_to_rgb(self, h, s, v):
        if s == 0.0:
            return v, v, v
        i = int(h * 6)
        f = (h * 6) - i
        p = v * (1 - s)
        q = v * (1 - s * f)
        t = v * (1 - s * (1 - f))
        i %= 6
        if i == 0: return v, t, p
        if i == 1: return q, v, p
        if i == 2: return p, v, t
        if i == 3: return p, q, v
        if i == 4: return t, p, v
        if i == 5: return v, p, q
